import React, { useState, useEffect, useMemo } from 'react';
import { X, Plus, Search, Check, AlertCircle, Share2, Download } from 'lucide-react';
import { motion, AnimatePresence } from 'framer-motion';
import { getItems, getCustomers, saveInvoice, saveCustomer, saveItem } from '../services/api';
import jsPDF from 'jspdf';
import 'jspdf-autotable';

const InvoiceForm = ({ onClose }) => {
    const [customer, setCustomer] = useState(null);
    const [selectedItems, setSelectedItems] = useState([]);
    const [isAddingItem, setIsAddingItem] = useState(false);
    const [isSelectingCustomer, setIsSelectingCustomer] = useState(false);
    const [isQuickAddingCustomer, setIsQuickAddingCustomer] = useState(false);
    const [isQuickAddingItem, setIsQuickAddingItem] = useState(false);

    const [itemsData, setItemsData] = useState([]);
    const [customersData, setCustomersData] = useState([]);
    const [saving, setSaving] = useState(false);

    // Search states
    const [customerSearch, setCustomerSearch] = useState('');
    const [itemSearch, setItemSearch] = useState('');

    // Quick Add Customer States
    const [newCustName, setNewCustName] = useState('');
    const [newKraPin, setNewKraPin] = useState('');
    const [kraError, setKraError] = useState('');

    // Quick Add Item States
    const [newItemName, setNewItemName] = useState('');
    const [newItemPrice, setNewItemPrice] = useState('');
    const [newItemUnit, setNewItemUnit] = useState('');
    const [newItemCategory, setNewItemCategory] = useState('');

    useEffect(() => {
        refreshData();
    }, []);

    const refreshData = () => {
        getItems().then(setItemsData);
        getCustomers().then(setCustomersData);
    };

    const validateKraPin = (pin) => {
        const regex = /^[AP][0-9]{9}[a-zA-Z]$/;
        return regex.test(pin);
    };

    const handleQuickAddCustomer = async () => {
        if (!newCustName) return alert('Enter customer name');
        if (!validateKraPin(newKraPin)) {
            setKraError('Invalid KRA PIN. Must be 11 chars (starts with A/P, ends with letter)');
            return;
        }
        setKraError('');
        setSaving(true);
        try {
            const resp = await saveCustomer({ name: newCustName, kra_pin: newKraPin });
            alert('Customer added!');
            setIsQuickAddingCustomer(false);
            setNewCustName('');
            setNewKraPin('');
            refreshData();
        } catch (err) {
            alert(err.message);
        } finally {
            setSaving(false);
        }
    };

    const handleQuickAddItem = async () => {
        if (!newItemName || !newItemPrice) return alert('Enter name and price');
        setSaving(true);
        try {
            await saveItem({
                name: newItemName,
                price: parseFloat(newItemPrice),
                unit: newItemUnit,
                category: newItemCategory
            });
            alert('Item added!');
            setIsQuickAddingItem(false);
            setNewItemName('');
            setNewItemPrice('');
            setNewItemUnit('');
            setNewItemCategory('');
            refreshData();
        } catch (err) {
            alert(err.message);
        } finally {
            setSaving(false);
        }
    };

    const filteredCustomers = useMemo(() => {
        return customersData.filter(c => c.name.toLowerCase().includes(customerSearch.toLowerCase()));
    }, [customersData, customerSearch]);

    const filteredItems = useMemo(() => {
        return itemsData.filter(i => i.name.toLowerCase().includes(itemSearch.toLowerCase()));
    }, [itemsData, itemSearch]);

    const addItem = (item) => {
        const existing = selectedItems.find(i => i.id === item.id);
        if (existing) {
            setSelectedItems(selectedItems.map(i => i.id === item.id ? { ...i, qty: i.qty + 1 } : i));
        } else {
            setSelectedItems([...selectedItems, { ...item, qty: 1 }]);
        }
    };

    const subtotal = selectedItems.reduce((acc, curr) => acc + (parseFloat(curr.price) * curr.qty), 0);
    const taxRate = 0.16; // KRA 16% VAT
    const taxAmount = subtotal * taxRate;
    const total = subtotal + taxAmount;

    const generatePDF = () => {
        const doc = new jsPDF();
        doc.setFontSize(20);
        doc.setTextColor(0, 119, 197); // Intuit Blue
        doc.text('INVOICE', 105, 20, { align: 'center' });

        doc.setFontSize(10);
        doc.setTextColor(0, 0, 0);
        doc.text(`Invoice Date: ${new Date().toLocaleDateString()}`, 150, 40);

        doc.setFontSize(12);
        doc.text('Bill To:', 20, 40);
        doc.text(customer?.name || 'Customer Name', 20, 50);
        if (customer?.kra_pin) doc.text(`KRA PIN: ${customer.kra_pin}`, 20, 60);

        const tableData = selectedItems.map(item => [
            item.name,
            item.qty,
            `$${parseFloat(item.price).toFixed(2)}`,
            `$${(item.qty * item.price).toFixed(2)}`
        ]);

        doc.autoTable({
            startY: 70,
            head: [['Description', 'Qty', 'Unit Price', 'Amount']],
            body: tableData,
            headStyles: { fillColor: [0, 119, 197] },
        });

        const finalY = (doc.lastAutoTable.finalY || 70) + 10;
        doc.text(`Subtotal: $${subtotal.toFixed(2)}`, 140, finalY);
        doc.text(`VAT (16%): $${taxAmount.toFixed(2)}`, 140, finalY + 10);
        doc.setFont(undefined, 'bold');
        doc.text(`Total: $${total.toFixed(2)}`, 140, finalY + 20);

        return doc;
    };

    const handleSharePDF = () => {
        if (!customer) return alert('Select a customer first');
        const doc = generatePDF();
        const pdfBlob = doc.output('blob');
        const file = new File([pdfBlob], `Invoice_${customer.name}.pdf`, { type: 'application/pdf' });

        if (navigator.share) {
            navigator.share({
                files: [file],
                title: 'Invoice',
                text: `Here is your invoice for $${total.toFixed(2)}`
            }).catch(console.error);
        } else {
            doc.save(`Invoice_${customer.name}.pdf`);
        }
    };

    const handleSaveInvoice = async () => {
        if (!customer) return alert('Please select a customer');
        if (selectedItems.length === 0) return alert('Please add at least one item');

        setSaving(true);
        try {
            await saveInvoice({
                customer_id: customer.id,
                subtotal: subtotal,
                tax_amount: taxAmount,
                total: total,
                items: selectedItems
            });
            alert('Invoice saved successfully!');
            onClose();
        } catch (err) {
            alert('Error saving invoice: ' + err.message);
        } finally {
            setSaving(false);
        }
    };

    return (
        <motion.div
            initial={{ y: '100%' }}
            animate={{ y: 0 }}
            exit={{ y: '100%' }}
            style={{
                position: 'fixed',
                top: 0,
                left: '50%',
                transform: 'translateX(-50%)',
                width: '100%',
                maxWidth: '500px',
                height: '100%',
                background: 'var(--bg-light)',
                zIndex: 2000,
                display: 'flex',
                flexDirection: 'column'
            }}
        >
            {/* Header */}
            <div style={{ padding: '16px', background: 'white', borderBottom: '1px solid var(--border-color)', display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
                <button onClick={onClose}><X size={24} /></button>
                <h2 style={{ fontSize: '18px' }}>New Invoice</h2>
                <button
                    onClick={handleSaveInvoice}
                    disabled={saving}
                    className="btn-primary"
                    style={{ padding: '6px 16px', fontSize: '14px', opacity: saving ? 0.7 : 1 }}
                >
                    {saving ? 'Saving...' : 'Save'}
                </button>
            </div>

            <div style={{ flex: 1, overflowY: 'auto', padding: '16px' }}>
                {/* Customer Selection */}
                <div className="card" onClick={() => setIsSelectingCustomer(true)}>
                    <p style={{ fontSize: '12px', color: 'var(--text-muted)' }}>Customer</p>
                    <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginTop: '4px' }}>
                        <div>
                            <h3 style={{ fontSize: '16px' }}>{customer ? customer.name : 'Select Customer'}</h3>
                            {customer?.kra_pin && <p style={{ fontSize: '12px', color: 'var(--text-muted)' }}>PIN: {customer.kra_pin}</p>}
                        </div>
                        <Plus size={20} color="var(--primary)" />
                    </div>
                </div>

                {/* Items List */}
                <h4 style={{ margin: '16px 0 8px' }}>Items</h4>
                {selectedItems.map(item => (
                    <div key={item.id} className="card" style={{ marginBottom: '8px', display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
                        <div>
                            <p style={{ fontWeight: 600 }}>{item.name}</p>
                            <p style={{ fontSize: '12px', color: 'var(--text-muted)' }}>${parseFloat(item.price).toFixed(2)} x {item.qty} {item.unit}</p>
                        </div>
                        <p style={{ fontWeight: 700 }}>${(parseFloat(item.price) * item.qty).toFixed(2)}</p>
                    </div>
                ))}

                <button
                    onClick={() => setIsAddingItem(true)}
                    style={{ width: '100%', padding: '12px', border: '1px dashed var(--primary)', color: 'var(--primary)', background: 'transparent', borderRadius: '8px', display: 'flex', alignItems: 'center', justifyContent: 'center', gap: '8px', marginBottom: '24px' }}
                >
                    <Plus size={20} /> Add Item
                </button>

                <div className="card" style={{ background: '#f8f9fa' }}>
                    <div style={{ display: 'flex', justifyContent: 'space-between', marginBottom: '8px' }}>
                        <p>Subtotal</p>
                        <p>${subtotal.toFixed(2)}</p>
                    </div>
                    <div style={{ display: 'flex', justifyContent: 'space-between', marginBottom: '8px', color: 'var(--text-muted)' }}>
                        <p>VAT (16% KRA)</p>
                        <p>${taxAmount.toFixed(2)}</p>
                    </div>
                    <div style={{ display: 'flex', justifyContent: 'space-between', fontWeight: 700, fontSize: '18px', borderTop: '1px solid var(--border-color)', paddingTop: '8px', marginTop: '8px' }}>
                        <p>Total</p>
                        <p>${total.toFixed(2)}</p>
                    </div>
                </div>

                <div style={{ marginTop: '24px', display: 'flex', gap: '12px' }}>
                    <button
                        onClick={handleSharePDF}
                        className="btn-secondary"
                        style={{ flex: 1, display: 'flex', alignItems: 'center', justifyContent: 'center', gap: '8px' }}
                    >
                        Share / Download PDF
                    </button>
                    <button
                        onClick={handleSaveInvoice}
                        disabled={saving}
                        className="btn-primary"
                        style={{ flex: 1, opacity: saving ? 0.7 : 1 }}
                    >
                        {saving ? 'Saving...' : 'Save Invoice'}
                    </button>
                </div>
            </div>

            {/* Selection Overlays */}
            <AnimatePresence>
                {/* Item Picker */}
                {isAddingItem && (
                    <motion.div
                        initial={{ opacity: 0, y: '100%' }}
                        animate={{ opacity: 1, y: 0 }}
                        exit={{ opacity: 0, y: '100%' }}
                        style={{ position: 'absolute', top: 0, left: 0, right: 0, bottom: 0, background: 'white', padding: '16px', zIndex: 2100, display: 'flex', flexDirection: 'column' }}
                    >
                        <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: '16px' }}>
                            <h3>Select Item</h3>
                            <button onClick={() => setIsAddingItem(false)}><X size={24} /></button>
                        </div>

                        <div style={{ position: 'relative', marginBottom: '16px' }}>
                            <Search style={{ position: 'absolute', left: '12px', top: '50%', transform: 'translateY(-50%)' }} size={18} color="var(--text-muted)" />
                            <input
                                className="search-input"
                                placeholder="Search items..."
                                value={itemSearch}
                                onChange={(e) => setItemSearch(e.target.value)}
                                style={{ width: '100%', padding: '10px 10px 10px 40px', borderRadius: '8px', border: '1px solid var(--border-color)' }}
                            />
                        </div>

                        <div style={{ flex: 1, overflowY: 'auto', display: 'flex', flexDirection: 'column', gap: '8px' }}>
                            {filteredItems.map(item => (
                                <div
                                    key={item.id}
                                    onClick={() => { addItem(item); setIsAddingItem(false); setItemSearch(''); }}
                                    style={{ padding: '12px', background: 'var(--bg-light)', borderRadius: '8px', display: 'flex', justifyContent: 'space-between', alignItems: 'center', cursor: 'pointer' }}
                                >
                                    <div>
                                        <p style={{ fontWeight: 500 }}>{item.name}</p>
                                        <p style={{ fontSize: '11px', color: 'var(--text-muted)' }}>{item.category} • {item.unit}</p>
                                    </div>
                                    <p style={{ color: 'var(--primary)', fontWeight: 600 }}>${parseFloat(item.price).toFixed(2)}</p>
                                </div>
                            ))}
                            <button
                                onClick={() => setIsQuickAddingItem(true)}
                                style={{ padding: '12px', background: 'transparent', border: '1px dashed var(--primary)', color: 'var(--primary)', borderRadius: '8px', marginTop: '10px' }}
                            >
                                + Quick Add New Item
                            </button>
                        </div>
                    </motion.div>
                )}

                {/* Customer Picker */}
                {isSelectingCustomer && (
                    <motion.div
                        initial={{ opacity: 0, y: '100%' }}
                        animate={{ opacity: 1, y: 0 }}
                        exit={{ opacity: 0, y: '100%' }}
                        style={{ position: 'absolute', top: 0, left: 0, right: 0, bottom: 0, background: 'white', padding: '16px', zIndex: 2100, display: 'flex', flexDirection: 'column' }}
                    >
                        <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: '16px' }}>
                            <h3>Select Customer</h3>
                            <button onClick={() => setIsSelectingCustomer(false)}><X size={24} /></button>
                        </div>

                        <div style={{ position: 'relative', marginBottom: '16px' }}>
                            <Search style={{ position: 'absolute', left: '12px', top: '50%', transform: 'translateY(-50%)' }} size={18} color="var(--text-muted)" />
                            <input
                                className="search-input"
                                placeholder="Search customers..."
                                value={customerSearch}
                                onChange={(e) => setCustomerSearch(e.target.value)}
                                style={{ width: '100%', padding: '10px 10px 10px 40px', borderRadius: '8px', border: '1px solid var(--border-color)' }}
                            />
                        </div>

                        <div style={{ flex: 1, overflowY: 'auto', display: 'flex', flexDirection: 'column', gap: '8px' }}>
                            {filteredCustomers.map(c => (
                                <div
                                    key={c.id}
                                    onClick={() => { setCustomer(c); setIsSelectingCustomer(false); setCustomerSearch(''); }}
                                    style={{ padding: '12px', background: 'var(--bg-light)', borderRadius: '8px', cursor: 'pointer' }}
                                >
                                    <p style={{ fontWeight: 500 }}>{c.name}</p>
                                    {c.kra_pin && <p style={{ fontSize: '11px', color: 'var(--text-muted)' }}>PIN: {c.kra_pin}</p>}
                                </div>
                            ))}
                            <button
                                onClick={() => setIsQuickAddingCustomer(true)}
                                style={{ padding: '12px', background: 'transparent', border: '1px dashed var(--primary)', color: 'var(--primary)', borderRadius: '8px', marginTop: '10px' }}
                            >
                                + Quick Add New Customer
                            </button>
                        </div>
                    </motion.div>
                )}

                {/* Quick Add Customer Form */}
                {isQuickAddingCustomer && (
                    <motion.div
                        initial={{ opacity: 0 }} animate={{ opacity: 1 }} exit={{ opacity: 0 }}
                        style={{ position: 'absolute', top: 0, left: 0, right: 0, bottom: 0, background: 'rgba(0,0,0,0.5)', zIndex: 2200, display: 'flex', alignItems: 'center', padding: '20px' }}
                    >
                        <div style={{ background: 'white', width: '100%', borderRadius: '16px', padding: '20px' }}>
                            <h3>Quick Add Customer</h3>
                            <div style={{ display: 'flex', flexDirection: 'column', gap: '12px', marginTop: '16px' }}>
                                <input placeholder="Name" value={newCustName} onChange={(e) => setNewCustName(e.target.value)} className="form-input" />
                                <input placeholder="KRA PIN (e.g. P123456789F)" value={newKraPin} onChange={(e) => setNewKraPin(e.target.value.toUpperCase())} className="form-input" />
                                {kraError && <p style={{ color: 'var(--error)', fontSize: '12px' }}>{kraError}</p>}
                                <div style={{ display: 'flex', gap: '8px', marginTop: '8px' }}>
                                    <button onClick={() => setIsQuickAddingCustomer(false)} className="btn-secondary" style={{ flex: 1 }}>Cancel</button>
                                    <button onClick={handleQuickAddCustomer} className="btn-primary" style={{ flex: 1 }}>Save</button>
                                </div>
                            </div>
                        </div>
                    </motion.div>
                )}

                {/* Quick Add Item Form */}
                {isQuickAddingItem && (
                    <motion.div
                        initial={{ opacity: 0 }} animate={{ opacity: 1 }} exit={{ opacity: 0 }}
                        style={{ position: 'absolute', top: 0, left: 0, right: 0, bottom: 0, background: 'rgba(0,0,0,0.5)', zIndex: 2200, display: 'flex', alignItems: 'center', padding: '20px' }}
                    >
                        <div style={{ background: 'white', width: '100%', borderRadius: '16px', padding: '20px' }}>
                            <h3>Quick Add Item</h3>
                            <div style={{ display: 'flex', flexDirection: 'column', gap: '12px', marginTop: '16px' }}>
                                <input placeholder="Item Name" value={newItemName} onChange={(e) => setNewItemName(e.target.value)} className="form-input" />
                                <input placeholder="Price" type="number" value={newItemPrice} onChange={(e) => setNewItemPrice(e.target.value)} className="form-input" />
                                <input placeholder="Unit (e.g. pcs, hrs)" value={newItemUnit} onChange={(e) => setNewItemUnit(e.target.value)} className="form-input" />
                                <input placeholder="Category" value={newItemCategory} onChange={(e) => setNewItemCategory(e.target.value)} className="form-input" />
                                <div style={{ display: 'flex', gap: '8px', marginTop: '8px' }}>
                                    <button onClick={() => setIsQuickAddingItem(false)} className="btn-secondary" style={{ flex: 1 }}>Cancel</button>
                                    <button onClick={handleQuickAddItem} className="btn-primary" style={{ flex: 1 }}>Save</button>
                                </div>
                            </div>
                        </div>
                    </motion.div>
                )}
            </AnimatePresence>
        </motion.div>
    );
};

export default InvoiceForm;
